<?php
/**
 * Created by vm.pl
 * User: Rafał Ignaszewski (rafal@vm.pl)
 * Date: 11.12.13 12:08
 */

namespace VM\AdminBundle\Services;

use Doctrine\ORM\EntityManager;
use Import\ApiBundle\Entity\AllegroCategoryItem;
use Import\ApiBundle\Entity\AllegroCategoryItemType;
use Import\ApiBundle\Entity\Wibor;
use Symfony\Component\Security\Core\SecurityContext;
use VM\ApiBundle\Entity\BankServiceOptions;
use VM\ApiBundle\Entity\Company;
use VM\ApiBundle\Entity\CompanyAllegroCategoryItem;
use VM\ApiBundle\Entity\CompanyBankServiceOption;
use VM\ApiBundle\Entity\CompanyReport;
use VM\ApiBundle\Entity\Department;
use VM\ApiBundle\Entity\DistributionSales;
use VM\ApiBundle\Entity\Employee;
use VM\ApiBundle\Entity\GroupRoundReportCode;
use VM\ApiBundle\Entity\MarketSegment;
use VM\ApiBundle\Entity\Offer;
use VM\ApiBundle\Entity\OfferReport;
use VM\ApiBundle\Entity\Round;
use VM\ApiBundle\Entity\User;
use VM\ApiBundle\Entity\UserGroups;

class ApiGetter
{
    /**
     * @var EntityManager
     */
    protected $EntityManager;

    /**
     * @var SecurityContext
     */
    protected $SecurityContext;


    public function __construct(EntityManager $EntityManager, SecurityContext $SecurityContext)
    {
        $this->EntityManager = $EntityManager;
        $this->SecurityContext = $SecurityContext;
    }


    /**
     * @return EntityManager
     */
    public function getEntityManager()
    {
        return $this->EntityManager;
    }


    /**
     * @param $groupId
     * @return UserGroups
     * @throws \Exception
     */
    public function getGroup($groupId)
    {
        $Group = $this->EntityManager->getRepository('VMApiBundle:UserGroups')->find($groupId);
        if(empty($Group))
        {
            throw new \Exception('Group not exists');
        }

        return $Group;
    }


    /**
     * @param $id
     * @return User
     */
    public function getUser($id)
    {
        $User = $this->EntityManager->getRepository('VMApiBundle:User')->find($id);
        if(empty($User))
        {
            throw new \Exception('User not exists');
        }

        return $User;
    }


    /**
     * @param $externalUserId
     * @return User
     * @throws \Exception
     */
    public function getUserByExternalId($externalUserId)
    {
        $User = $this->EntityManager->getRepository('VMApiBundle:User')->findOneBy(array('external_id' => $externalUserId));
        if(empty($User))
        {
            throw new \Exception('User not exists');
        }

        return $User;
    }


    /**
     * @param $externalUserId
     * @return User
     */
    public function getUserByEmail($email)
    {
        $User = $this->EntityManager->getRepository('VMApiBundle:User')->findOneByEmail($email);
        if(empty($User))
        {
            throw new \Exception('User not exists');
        }

        return $User;
    }


    /**
     * @param User $User
     * @param UserGroups $Group
     * @return bool
     */
    public function isUserGroup(User $User, UserGroups $Group)
    {
        return $User->getId() == $Group->getUser()->getId();
    }


    /**
     * @param $id
     * @return Round
     */
    public function getRound($id)
    {
        $Round = $this->EntityManager->getRepository('VMApiBundle:Round')->find($id);
        if(empty($Round))
        {
            throw new \Exception('Round not exists');
        }

        return $Round;
    }


    /**
     * Zwraca obiekt aktualnej rundy dla uzytkownika lub false jeśli nie zostały ustawione lub się jeszcze nie rozpoczęły
     *
     * @param User $User
     * @return Round
     * @throws \Exception
     */
    public function getUserActualRound(User $User)
    {
        if($this->isTeacher($User))
        {
            throw new \Exception('Teacher is not member of any group (UserActualRound)');
        }
        else
        {
            $rounds = $User->getGroup()->getRounds();
        }

        if(count($rounds) == 0)
        {
            throw new \Exception('Rounds are not set properly', 40000);
        }

        $CurrentRound = false;
        $Now = new \DateTime();

        /** @var Round $Round */
        foreach($rounds as $Round)
        {
            $StartAt = $Round->getStartAt();
            if(!empty($StartAt) && $StartAt->format("YmdHis") <= $Now->format("YmdHis") && !$Round->getIsDone())
            {
                $CurrentRound = $Round;
                break;
            }
        }

        return $CurrentRound;
    }

    /**
     * Zwraca obiekt aktualnej rundy dla grupy lub false jeśli nie zostały ustawione lub się jeszcze nie rozpoczęły
     *
     * @param UserGroups $Group
     * @return Round
     * @throws \Exception
     */
    public function getGroupActualRound(UserGroups $Group)
    {
        $rounds = $Group->getRounds();

        if(count($rounds) == 0)
        {
            throw new \Exception('Rounds are not set properly', 40000);
        }

        $CurrentRound = false;
        $Now = new \DateTime();

        /** @var Round $Round */
        foreach($rounds as $Round)
        {
            $StartAt = $Round->getStartAt();
            if(!empty($StartAt) && $StartAt->format("YmdHis") <= $Now->format("YmdHis") && !$Round->getIsDone())
            {
                $CurrentRound = $Round;
                break;
            }
        }

        return $CurrentRound;
    }

    public function getNextRound(User $User, Round $ActualRound)
    {
        if($this->isTeacher($User))
        {
            throw new \Exception('Teacher is not member of any group (NextRound)');
        }
        else
        {
            $rounds = $User->getGroup()->getRounds();
        }
        if(count($rounds) == 0)
        {
            throw new \Exception('Rounds are not set properly', 40000);
        }
        $seq = $ActualRound->getSequence();
        $seq++;

        return $rounds->filter(function($Round) use($seq){return $Round->getSequence() == $seq;})->first();
    }


    /**
     * Zwraca firmę uzytkownika
     *
     * @param User $User
     * @return Company
     */
    public function getUserCompanyOrFalse(User $User)
    {
        return $this->getGroupCompanyOrFalse($User->getGroup());
    }


    /**
     * @param UserGroups $Group
     * @return Company
     */
    public function getGroupCompanyOrFalse(UserGroups $Group)
    {
        return $Group->getCompany()->first();
    }


    /**
     * @param $departmentId
     * @return Department
     * @throws \Exception
     */
    public function getDepartment($departmentId)
    {
        $Department = $this->EntityManager->getRepository('VMApiBundle:Department')->find($departmentId);
        if(empty($Department))
        {
            throw new \Exception('Department not exists');
        }
        return $Department;
    }


    /**
     * @param $employeeId
     * @return Employee
     * @throws \Exception
     */
    public function getEmployee($employeeId)
    {
        $Employee = $this->EntityManager->getRepository('VMApiBundle:Employee')->find($employeeId);
        if(empty($Employee))
        {
            throw new \Exception('Employee not exists');
        }
        return $Employee;
    }


    /**
     * @param $id
     * @return MarketSegment
     * @throws \Exception
     */
    public function getMarketSegment($id)
    {
        $MarketSegment = $this->EntityManager->getRepository('VMApiBundle:MarketSegment')->find($id);
        if(empty($MarketSegment))
        {
            throw new \Exception('MarketSegment not exists');
        }

        return $MarketSegment;
    }

    /**
     * @param User $User
     * @return MarketSegment
     */
    public function getMarketSegmentForUser(User $User)
    {
        return $User->getGroup()->getUser()->getmarketSegment();
    }


    /**
     * @param $id
     * @return AllegroCategoryItem
     * @throws \Exception
     */
    public function getAllegroItem($id)
    {
        $AllegroCategoryItem = $this->EntityManager->getRepository('ImportApiBundle:AllegroCategoryItem')->find($id);
        if(empty($AllegroCategoryItem))
        {
            throw new \Exception('AllegroCategoryItem not exists');
        }

        return $AllegroCategoryItem;
    }


    /**
     * @param $id
     * @return CompanyAllegroCategoryItem
     * @throws \Exception
     */
    public function getCompanyAllegroItem($id)
    {
        $CompanyAllegroCategoryItem = $this->EntityManager->getRepository('VMApiBundle:CompanyAllegroCategoryItem')->find($id);
        if(empty($CompanyAllegroCategoryItem))
        {
            throw new \Exception('CompanyAllegroCategoryItem not exists');
        }

        return $CompanyAllegroCategoryItem;
    }


    /**
     * @param User $User
     * @return bool
     */
    public function isTeacher(User $User)
    {
        return (array_search('ROLE_TEACHER', $User->getRoles()) !== false);
//        return $this->SecurityContext->isGranted('ROLE_TEACHER');
    }


    /**
     * @param Round $Round
     * @return bool|Round
     */
    public function getNextRoundOrFalse(Round $Round)
    {
        $NextRound = $this->EntityManager->getRepository('VMApiBundle:Round')->findOneBy(array(
            'userGroupId' => $Round->getUserGroupId(),
            'sequence' => $Round->getSequence() + 1
        ));
        if(empty($NextRound))
        {
            return false;
        }
        return $NextRound;
    }


    /**
     * @param Round $Round
     * @return bool|Round
     */
    public function getPreviousRoundOrFalse(Round $Round)
    {
        $PrevRound = $this->EntityManager->getRepository('VMApiBundle:Round')->findOneBy(array(
            'userGroupId' => $Round->getUserGroupId(),
            'sequence' => $Round->getSequence() - 1
        ));
        if(empty($PrevRound))
        {
            return false;
        }
        return $PrevRound;
    }


    /**
     * @param $code
     * @return CompanyReport
     * @throws \Exception
     */
    public function getCompanyReportByCode($code)
    {
        $code = $this->EntityManager->getRepository('VMApiBundle:GroupRoundReportCode')->findOneByCode($code);
        if(empty($code))
        {
            throw new \Exception('Code not exists');
        }
        $CompanyReport = $this->EntityManager->getRepository('VMApiBundle:CompanyReport')->findOneByCodeId($code->getId());
        if(empty($CompanyReport))
        {
            throw new \Exception('Report not exists');
        }

        return $CompanyReport;
    }


    /**
     * @param Round $Round
     * @param Company $Company
     * @param bool $extendedReport
     * @return bool|CompanyReport
     */
    public function getCompanyReportOrFalse(Round $Round, Company $Company, $extendedReport = false)
    {
        $CompanyReport = $this->EntityManager->getRepository('VMApiBundle:CompanyReport')->findOneBy(array(
            'roundId' => $Round->getId(),
            'companyId' => $Company->getId(),
            'extended' => $extendedReport
        ));
        if(empty($CompanyReport))
        {
            return false;
        }

        return $CompanyReport;
    }


    /**
     * @param Company $Company
     * @return mixed
     */
    public function getCompanyBankServiceOptions(Company $Company)
    {
        return $this->EntityManager->getRepository('VMApiBundle:BankService')->getCompanyBankServiceOptions($Company->getId());
    }


    /**
     * @param $optionId
     * @return BankServiceOptions
     * @throws \Exception
     */
    public function getBankServiceOption($optionId)
    {
        $BankServiceOption = $this->EntityManager->getRepository('VMApiBundle:BankServiceOptions')->find($optionId);
        if(empty($BankServiceOption))
        {
            throw new \Exception('BankServiceOption not exist');
        }

        return $BankServiceOption;
    }


    /**
     * @param $companyBankServiceOptionId
     * @return CompanyBankServiceOption
     * @throws \Exception
     */
    public function getCompanyBankServiceOption($companyBankServiceOptionId)
    {
        $CompanyBankServiceOption = $this->EntityManager->getRepository('VMApiBundle:CompanyBankServiceOption')->find($companyBankServiceOptionId);
        if(empty($CompanyBankServiceOption))
        {
            throw new \Exception('CompanyBankServiceOption not exist');
        }

        return $CompanyBankServiceOption;
    }


    /**
     * @param $companyId
     * @return CompanyBankServiceOption|null
     */
    public function getCompanyCurrentAccount($companyId)
    {
        return $this->EntityManager->getRepository('VMApiBundle:CompanyBankServiceOption')->getCompanyCurrentAccountOption($companyId);
    }


    /**
     * @param $offerReportId
     * @return OfferReport
     * @throws \Exception
     */
    public function getOfferReport($offerReportId)
    {
        $OfferReport = $this->EntityManager->getRepository('VMApiBundle:OfferReport')->find($offerReportId);
        if(empty($OfferReport))
        {
            throw new \Exception('OfferReport not exist');
        }

        return $OfferReport;
    }

    /**
     * @param $offerId
     * @return Offer
     * @throws \Exception
     */
    public function getOffer($offerId)
    {
        $Offer = $this->EntityManager->getRepository('VMApiBundle:Offer')->find($offerId);
        if(empty($Offer))
        {
            throw new \Exception('Offer not exist');
        }

        return $Offer;
    }

    /**
     * @param $distributionSalesId
     * @return DistributionSales
     * @throws \Exception
     */
    public function getDistributionSales($distributionSalesId)
    {
        $DistributionSales = $this->EntityManager->getRepository('VMApiBundle:DistributionSales')->find($distributionSalesId);
        if(empty($DistributionSales))
        {
            throw new \Exception('DistributionSales not exist');
        }

        return $DistributionSales;
    }

    /**
     * @param $symbol
     * @return AllegroCategoryItemType
     * @throws \Exception
     */
    public function getAllegroCategoryItemTypeBySymbol($symbol)
    {
        $AllegroCategoryItemType = $this->EntityManager->getRepository('ImportApiBundle:AllegroCategoryItemType')->findOneBy(array('symbol'=>$symbol));
        if(empty($AllegroCategoryItemType))
        {
            throw new \Exception('AllegroCategoryItemType not exist');
        }

        return $AllegroCategoryItemType;
    }

    /**
     * @return float|int
     */
    public function getActualWiborValue()
    {
        /** @var Wibor $Wibor */
        $Wibor = $this->EntityManager->getRepository('Import\ApiBundle\Entity\Wibor')->getActualWibor();
        if(empty($Wibor))
        {
            return 0;
        }
        else
        {
            return $Wibor->getValue();
        }
    }

    /**
     * @param Company $Company
     * @param Round $Round
     * @return bool|GroupRoundReportCode
     */
    public function getAuthProductCodeForCompanyAndRoundOrFalse(Company $Company, Round $Round, $extended = false)
    {
        /** @var GroupRoundReportCode $GroupRoundReportCode */
        $GroupRoundReportCode =
            $this->EntityManager->getRepository('VMApiBundle:GroupRoundReportCode')
            ->findOneBy(array('round' => $Round, 'group' => $Company->getUserGroups(), 'extended' => $extended));

        return empty($GroupRoundReportCode)? false: $GroupRoundReportCode;
    }

}